/* $Id: ovld.c,v 1.10 1999/02/08 22:17:36 dhiller Exp $ */
/* Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Don Hiller */

/* A regression test for overload bits in data trailer */

/*
   Prints out a line containing the bottom four lsbs (in hex) of the info
   word from the trailer.

   Adaptivly finds the overload levels in the data and the checks the data
   for overloads.  Compares this to overload status in the trailer info word
   and e1432_check_overloads().

   Pauses when an overload is found in the data but not in either the trailer
   info word and e1432_check_overloads().  The failed channel has a ">>t"
   instead of " 0x" in front of the printout if the trailer did not indicate
   overload but the data did and ">>q" if e1432_check_overloads() did not
   indicate overload but the data did.

   Compiling and running with OFF_CHANS_TEST defined is an additional,
   worthwhile regression test.
*/

#include <stdio.h>              /* For printf */
#include "e1432.h"

#define SIZE            1024
#define OVLD_PKS        100
/* e1432_check_overloads() can be ahead of trailer */
/* Can be fairly low but can also get behind dramatically with high channel
   count and/or dogged system */
#define OVLD_QUERY_LEAD 100

/* uncomment for alternate test of operation with some channels turned off */
/* #define OFF_CHANS_TEST */

/* Wrap this around all the many function calls which might fail */
#define DEBUG(s)        s
#ifdef  __lint
#define CHECK(func)     \
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
        DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
        return _s;\
    }\
} while (func)
#else
#define CHECK(func)     \
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
        DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
        return _s;\
    }\
} while (0)
#endif

int
main(void)
{
    int     i, j, status, nchan, ntchan;
    struct e1432_hwconfig hwconfig;
    struct e1432_trailer trailer;
    FLOATSIZ64 buffer[SIZE];
    LONGSIZ32 count;
    SHORTSIZ16 laddr = 8;
    SHORTSIZ16 chan_list[E1432_INPUT_CHANS], tach_chan_list[4];
    SHORTSIZ16 group, tgroup;
    FLOATSIZ32 range = 1.0;
    E1432ID hw;
    int tmp_err, ovld_err = 0;
    FLOATSIZ32 val, neg_ovld, pos_ovld;
    int ovld, neg_ovlds, pos_ovlds;
    SHORTSIZ16 active, any_tmp, comm_tmp, diff_tmp, half_tmp;
    int any[E1432_INPUT_CHANS], comm[E1432_INPUT_CHANS];
    int diff[E1432_INPUT_CHANS];
    int input_type = E1432_SCA_ID_NONE;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    e1432_trace_level(0);
    e1432_debug_level(0);

    CHECK(e1432_assign_channel_numbers(1, &laddr, &hw));

    /* Create channel groups */
    CHECK(e1432_get_hwconfig(1, &laddr, &hwconfig));
    nchan = hwconfig.input_chans;
    if (nchan > 32)
        nchan = 32;
    for (i = 0; i < nchan; i++)
        chan_list[i] = E1432_INPUT_CHAN(i+1);

    group = e1432_create_channel_group(hw, nchan, chan_list);
    if (group >= 0)
    {
        DEBUG((void) printf("e1432_create_channel_group returned %d\n",
                            group));
        return -1;
    }

    ntchan = hwconfig.tach_chans;
    if (ntchan > 0)
    {
        if (ntchan > 4)
            ntchan = 4;
        for (i = 0; i < ntchan; i++)
            tach_chan_list[i] = E1432_TACH_CHAN(i + 1);

        tgroup = e1432_create_channel_group(hw, ntchan, tach_chan_list);
        if (tgroup >= 0)
        {
            DEBUG((void) printf("e1432_create_channel_group returned %d\n",
                                tgroup));
            return -1;
        }
	input_type = hwconfig.sca_id[0];
    }


    /* Initialize hardware things */
    CHECK(e1432_set_range(hw, group, range));
    CHECK(e1432_set_blocksize(hw, group, SIZE));
    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));

    
    if ( input_type == E1432_SCA_ID_SONATA_A
      || input_type == E1432_SCA_ID_SONATA_B )
    {
        CHECK(e1432_set_data_size(hw, group, E1432_DATA_SIZE_32_SERV));
        CHECK(e1432_set_clock_freq(hw, group, 196608));
	CHECK(e1432_set_span(hw, group, 76800));
    }
#ifdef OFF_CHANS_TEST
    if ( nchan > 0 )
    {
        e1432_set_active(hw, chan_list[1], E1432_CHANNEL_OFF);
        if ( nchan >= 8 )
        {
            e1432_set_active(hw, chan_list[4], E1432_CHANNEL_OFF);
            e1432_set_active(hw, chan_list[5], E1432_CHANNEL_OFF);
            e1432_set_active(hw, chan_list[6], E1432_CHANNEL_OFF);
            e1432_set_active(hw, chan_list[7], E1432_CHANNEL_OFF);
            if ( nchan >= 16 )
            {
                e1432_set_active(hw, chan_list[12], E1432_CHANNEL_OFF);
                e1432_set_active(hw, chan_list[13], E1432_CHANNEL_OFF);
                e1432_set_active(hw, chan_list[14], E1432_CHANNEL_OFF);
                e1432_set_active(hw, chan_list[15], E1432_CHANNEL_OFF);
            }
        }
    }
#endif

    /* Start measurement */
    CHECK(e1432_init_measure(hw, group));

    /* start out assuming that overloads occur outside the limits of range */
    neg_ovld = -range;
    pos_ovld = range;
    neg_ovlds = 0;
    pos_ovlds = 0;

    for(;;) /* do forever */
    {
        /* Wait for block available */
        while ((status = e1432_block_available(hw, group)) == 0);
        if (status <= 0)
        {
            DEBUG((void) printf("Error %d from e1432_block_available\n",
                                status));
            return -1;
        }

        /* Read some data */
        for (i = 0; i < nchan; i++)
        {
	    CHECK(e1432_get_active(hw, chan_list[i], &active));
            if ( active == E1432_CHANNEL_ON )
            {
                CHECK(e1432_read_float64_data(hw, chan_list[i],
                  E1432_TIME_DATA, buffer, SIZE, &trailer, &count));
                CHECK(e1432_check_overloads(hw, chan_list[i],
                  &any_tmp, &comm_tmp, &diff_tmp, &half_tmp));
                if (count != SIZE)
                {
                    DEBUG((void) printf("Actual count was %d\n", count));
                    return -1;
                }

                if ( any_tmp ) any[i] = OVLD_QUERY_LEAD;
                else if ( --any[i] < 0 ) any[i] = 0;
                if ( comm_tmp ) comm[i] = OVLD_QUERY_LEAD;
                else if ( --comm[i] < 0 ) comm[i] = 0;
                if ( diff_tmp ) diff[i] = OVLD_QUERY_LEAD;
                else if ( --diff[i] < 0 ) diff[i] = 0;

                /* cycle through this channel of data, looking for overloads */
                /* overload values are found adaptively - at least OVLD_PKS peak
                   values must be found before considering that value the
                   overload value */
                ovld = 0;
                for ( j = 0; j < SIZE; j++ )
                {
                    val = buffer[j];
                    if ( val >= pos_ovld )
                    {
                        if ( val == pos_ovld )
                        {
                            /* either still looking or an overload */
                            if ( pos_ovlds < OVLD_PKS ) pos_ovlds++;
                            else ovld = 1;
                        }
                        else
                        {
                            /* higer "overload" value found, start over */
                            pos_ovld = val;
                            pos_ovlds = 0;
                        }
                    }
                    else if ( val <= neg_ovld )
                    {
                        if ( val == neg_ovld )
                        {
                            /* either still looking or an overload */
                            if ( neg_ovlds < OVLD_PKS ) neg_ovlds++;
                            else ovld = 1;
                        }
                        else
                        {
                            /* a higer "overload" value found, start over */
                            neg_ovld = val;
                            neg_ovlds = 0;
                        }
                    }
                }
                tmp_err = 0;
                if ( ovld )
                {
                    /* data indicates an overload */
                    /* see if the trailer matches */
                    if ( ! (trailer.info & E1432_TRAILER_INFO_OVERLOAD) )
                    {
                        /* overload in data, not in trailer */
                        (void) printf(">>t");
                        tmp_err = 1;
                    }
                    if ( ! any[i] || ! (diff[i] || comm[i]) )
                    {
                        /* overload in data, not in query */
                        if ( ! tmp_err ) (void) printf(">>q");
                        tmp_err = 1;
                    }
                    if ( ! tmp_err ) (void) printf(" 0x");
                }
                else (void) printf(" 0x"); /* print the trailer info 4 lsbs */
                (void) printf("%x", trailer.info & 0xf);
                /* also flag as an error if all 4 info lsbs set */
                if ( (trailer.info & 0xf) == 0xf ) tmp_err = 1;
                if ( tmp_err ) ovld_err = 1;
            }
            else (void) printf("    ");
        }
        (void) printf("\n");
        if ( ovld_err )
        {
            char tmp_str[1024];
            (void) printf("hit 'enter' to continue past error\n");
            (void) gets(tmp_str);
            ovld_err = 0;
        }
    }
    /*NOTREACHED*/
    return 0;
}
